/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  Image.h - FW image manipulation class
 *
 *  Version: $Id: Image.h 2752 2006-01-19 14:40:17Z mst $
 *
 */

#ifndef IMAGE_H
#define IMAGE_H

#include <list>
#include <map>
#include <string>
#include <vector>
#include <memory>

#include "Special.h"
#include "CrSpace.h"
#include "compatibility.h"
#include "utils.h"
#include "MultiFile.h"

//------------------------------------------
//------------------------------ Crc16 class
//------------------------------------------
//
class Crc16
{
public:
    Crc16(bool d = false) : _debug(d) { clear();       }
    u_int16_t      get()              { return _crc;   }
    void           clear()            { _crc = 0xffff; }
    void           operator<<(u_int32_t val) { add(val); }
    void           add(u_int32_t val);
    void           finish();
private:
    u_int16_t      _crc;
    bool           _debug;
};


//------------------------------------------
//------------------------------ Image class
//------------------------------------------
//
class Burn;
class ParamList;
class MultiFile;
class Image : public ErrMsg
{
public:
    Image(ParamList *plist , bool only_rmw_cr = false);
    virtual ~Image();

    // Output image formats
    enum FORMATS {
        OUT_IMAGE,      // Mellanox image format - byte after byte, 16 bytes in line
        OUT_DWORDS,     // Dword after dword, in separate line each
        OUT_BINARY      // RAW binary format
    };

    // Special STRUCT ID
    enum STRUCT_ID {
        SYS_GUID_ID  = 1,
        NODE_GUID_ID = 5,
        BSN_ID       = 6
    };

    // Parameters to save
    class Saved_params {
    public:
        std::string name;
        std::string value;
        u_int32_t   geo;
    };

    // FW Version Format: {Major,Minior,SubMinor}
    typedef   u_int32_t FwVer[3];

    virtual bool        param_check (ParamList& plist);
    virtual bool        preproc     (ParamList& plist, const MultiFile& mfile);
    virtual bool        compile     (ParamList& plist, const MultiFile& mfile);
    virtual bool        write       (const char *fname, const char *format_name, DeviceType dev_type);

    virtual bool        get_fw_ver  (FwVer v);
    
    //virtual bool        read(const char *dev, std::list<Saved_params>& pars, const int first_ee);

    static bool fs_image;  // Is image need be FailSafe


protected:
    // Data tag
    struct DATA_TAG
    {
        DATA_TAG()        { reset(); }
        void      reset() { geo_id=0xffffffff; length=0; RESERVED=type=0; }
        u_int32_t geo_id;
#if __BYTE_ORDER ==  __BIG_ENDIAN
        u_int8_t  type;
        u_int8_t  RESERVED;
        u_int16_t length;
#elif __BYTE_ORDER == __LITTLE_ENDIAN
        u_int16_t length;
        u_int8_t  RESERVED;
        u_int8_t  type;
#else
#error Wrong enianess
#endif
    };

    // Data initialization types
    enum {
        TYPE_IWI = 1,   // Individual words initialization
        TYPE_NBI = 2,   // Non-homogeneous block initialization
        TYPE_HBI = 3,   // Homogeneous block initialization
        TYPE_RMW = 4,   // Read-modify-write block initialization
        TYPE_JMP = 5,   // Jump address initialization
        TYPE_COD = 97,  // Code. Same as TYPE_NBI
        TYPE_SPC = 98,  // Special structire
        TYPE_LST = 99   // Last data record
    };

protected:

    class ImSection : public ErrMsg {
    public:
        ImSection(const std::string& name) : _name(name),  _rom_start_addr(0) {};
        virtual ~ImSection() {}

        const   std::string&    GetName  ()                  { return _name; }

        virtual bool       Pack(std::vector<u_int8_t>& data) = 0;
        virtual int        HdrSize()                    = 0;
        virtual int        Size()                       { return _data.size() + HdrSize() + 4;} //crc 
        virtual bool       SetNext(u_int32_t nptr)      { nptr = 0; return errmsg("Internal Error: ImSection::SetNext() not defined for sector %s", _name.c_str());}
        virtual u_int32_t  GetRomStartAddr()            { return _rom_start_addr;}
        virtual void       SetRomStartAddr(u_int32_t a) { _rom_start_addr = a;}
        
        bool               SetWord(u_int32_t byte_addr, u_int16_t data);
        bool               CalcCrc();
        u_int32_t          GetCrc();


    protected:
        const std::string                     _name;
        std::vector<u_int8_t>            _data;
        u_int32_t                        _rom_start_addr;
        Crc16                            _crc;
    };


    class NVRam : public ErrMsg
    {
    public:
        NVRam()  { clear_crc(); }

        bool     init(ParamList* plist);
        void     clear_crc()                     { _crc.clear();            }
        unsigned size()                          { return _data.size();     }
        void     operator+=(const u_int32_t val) { append(val);             }
        void     operator<<(const u_int32_t val) { append(val);             }
        void     finish()                        { _prev_addr = 0xfffffffe; }
        u_int8_t operator[](const u_int32_t addr){ return get(addr);        }
        u_int8_t get(const u_int32_t addr)       { return _data[addr];      }
        void     copy(const u_int32_t offs, const u_int32_t len, u_int8_t *buf)
                 { std::copy(&_data[offs], &_data[offs+len], buf); }
        u_int8_t get(const u_int32_t addr, std::string& sname);
        void     clear();
        void     start(const char *name);
        void     add_crc();
        void     append(const u_int32_t val);
        void     add(const u_int32_t addr, const u_int32_t val);
        void     pad_to_addr(u_int32_t addr);

        std::vector<u_int8_t>&
                 data()                          { return _data;}
        
        bool      _is_flash;
        u_int32_t _sect_size; // for flash
        u_int32_t _bank_size;
        u_int32_t _nbanks;


    private:
        u_int32_t                        _prev_addr;
        std::vector<u_int8_t>            _data;
        Crc16                            _crc;
        std::map<u_int32_t, std::string> _sects;
        std::map<u_int32_t, std::string>::iterator _isects;

        static const int                 _fsizes[];

    };

    bool       format        (ParamList& plist, const MultiFile& mfile);
    bool       dump          (FILE *fp, const char *fname, const unsigned int idx, const FORMATS frm);
    bool       dump_info     (FILE *fp, const Image::FORMATS frm, DeviceType dev_type);
    
    virtual
       bool    fs_info_dump  (FILE* fp);
    
    void       raw_dump      (FILE *fp, const unsigned int idx);
    
    virtual
       bool    write_image(FILE* fp, const char* fname, FORMATS frm);

    virtual
       bool    mlx_dump      (FILE *fp, const unsigned int idx);
    bool       dwords_dump   (FILE *fp, const unsigned int idx);
    bool       raw_burn      (Burn& b, const unsigned int idx, const bool ver = false);
    bool       binary_dump   (const char *basename, const unsigned int idx);
    u_int32_t  getOffs       (const u_int32_t addr);
    u_int8_t   getI2C        (const u_int32_t addr);

    // PLLs
    static const int  PLLS = 4;
    static const int  PLL_CSUM_OFFSET = 12;  
    u_int32_t         _pll[PLLS];

    // EEPROM order info
    static const int  IN_LINE = 4;
    static const int  EEPROMS = 8;

    int               _ee_amount;
    u_int32_t         _ee_info[EEPROMS];

    // Executables and jump addresses
    typedef std::pair<std::string, u_int32_t> Exe;  // Exe-name, exe-jump address
    Exe                          _boot;
    std::list<Exe>               _exes;

    // Cr-Spaces (per GEO ID)
    std::map<u_int32_t, CrSpace> _cr;
    std::map<u_int32_t, CrSpace> _cr_boot2;

    // Special structures (per GEO ID)
    std::map<u_int32_t, Special> _spec;

    // EEPROM image (first half of it, actually). The second half is identical
    NVRam                       _nvram;

    // CRC,GUIDs, BSN replacement info
    typedef std::pair<u_int32_t, u_int8_t> EE_loc; // EEPROM location (ADDR/I2C)
    class CRC_Replacement {
    public:
        CRC_Replacement(const EE_loc& f, const EE_loc& t, const EE_loc& l)
            : from(f), to(t), loc(l) {}
        EE_loc from;  // begin of CRC-protected section
        EE_loc to;    // end of CRC-protected
        EE_loc loc;   // CRC itself location
    };
    std::list<CRC_Replacement> _crc_repl;
    std::list<EE_loc>          _guid_repl;
    std::list<EE_loc>          _guids_repl;
    std::list<EE_loc>          _sysguid_repl;
    std::list<EE_loc>          _bsn_repl;
    std::list<EE_loc>          _psid_repl;
    std::list<EE_loc>          _vsd_repl;

    ParamList                  *_plist;

    bool                       _only_rmw_cr;

    u_int32_t                  _first_image_start;  // First image addr in rom
    u_int32_t                  _second_image_start; // Second image addr in rom (for failsafe compilation

    // GUID replacement info
    // BSN replacement info
};

#endif
